# generate_structure_page.py

import pandas as pd
import json
from pathlib import Path
from jinja2 import Environment, FileSystemLoader
import multiprocessing
import os
import math
import shutil 

# --- Konfigurācija ---
CURRENT_DIR = Path(__file__).parent
OUTPUT_DIR = CURRENT_DIR / "out"

# IZMAIŅA: Datu mape tagad saucas 'csv' un atrodas skripta mapē
DATA_DIR = CURRENT_DIR / "csv"

# Assets ceļi
ASSETS_SRC_DIR = CURRENT_DIR / "assets" 
ASSETS_DEST_DIR = OUTPUT_DIR / "assets"

worker_data = {}

def load_data():
    """Ielādē nepieciešamos CSV failus no 'csv' direktorijas."""
    print(f"Meklē datus mapē: {DATA_DIR}")
    print("Ielādē datus...")
    try:
        dtype_spec = {
            'regcode': str, 'closed': str,
            'at_legal_entity_registration_number': str,
            'legal_entity_registration_number': str,
            'id': str, 'statement_id': str,
            'name': str
        }
        
        register_df = pd.read_csv(DATA_DIR / 'register.csv', sep=';', low_memory=False, dtype=dtype_spec)
        members_df = pd.read_csv(DATA_DIR / 'members.csv', sep=';', low_memory=False, dtype=dtype_spec)
        financial_statements_df = pd.read_csv(DATA_DIR / 'financial_statements.csv', sep=';', low_memory=False, dtype=dtype_spec)
        income_statements_df = pd.read_csv(DATA_DIR / 'income_statements.csv', sep=';', low_memory=False, dtype=dtype_spec)
        
        print("Dati veiksmīgi ielādēti.")
        return register_df, members_df, financial_statements_df, income_statements_df
    except FileNotFoundError as e:
        print(f"KĻŪDA: Fails nav atrasts - {e}. Pārliecinies, ka CSV faili atrodas '{DATA_DIR}' direktorijā.")
        return None, None, None, None

def init_worker(fs_df, is_df):
    worker_data['fs_df'] = fs_df
    worker_data['is_df'] = is_df

def get_latest_profit_worker(regcode):
    fs_df = worker_data['fs_df']
    is_df = worker_data['is_df']
    company_fs = fs_df[fs_df['legal_entity_registration_number'] == regcode].copy()
    if company_fs.empty: return regcode, None
    company_fs['year'] = pd.to_numeric(company_fs['year'], errors='coerce')
    company_fs.dropna(subset=['year'], inplace=True)
    if company_fs.empty: return regcode, None
    latest_year = company_fs['year'].max()
    latest_year_statements = company_fs[company_fs['year'] == latest_year]
    ugp_statement = latest_year_statements[latest_year_statements['source_type'] == 'UGP']
    chosen_fs_row = None
    if not ugp_statement.empty: chosen_fs_row = ugp_statement.iloc[0]
    elif not latest_year_statements.empty: chosen_fs_row = latest_year_statements.iloc[0]
    else: return regcode, None
    statement_id = chosen_fs_row.get('id')
    if pd.isna(statement_id): return regcode, None
    income_statement = is_df[is_df['statement_id'] == str(statement_id)]
    if not income_statement.empty:
        profit_str = income_statement.iloc[0].get('net_income')
        try:
            raw_profit = float(str(profit_str).replace(',', '.'))
            rounding_factor = 1
            rounded_to = str(chosen_fs_row.get('rounded_to_nearest', 'ONES')).strip().upper()
            if rounded_to == 'THOUSANDS': rounding_factor = 1000
            elif rounded_to == 'MILLIONS': rounding_factor = 1000000
            final_profit = raw_profit * rounding_factor
            return regcode, final_profit
        except (ValueError, TypeError): return regcode, None
    return regcode, None

def main():
    # Pārbauda, vai eksistē izejas mape
    if not DATA_DIR.exists():
        print(f"KĻŪDA: Datu mape '{DATA_DIR}' neeksistē!")
        return

    register_df, members_df, fs_df, is_df = load_data()
    if register_df is None: return

    print("Atlasa aktīvos uzņēmumus...")
    terminated_str = register_df['terminated'].astype(str).str.strip()
    closed_str = register_df['closed'].astype(str).str.strip()
    term_mask = terminated_str.isin(['', 'nan', '0000-00-00']) | terminated_str.isnull()
    closed_mask = closed_str.isin(['', 'nan']) | closed_str.isnull()
    active_companies_mask = term_mask & closed_mask
    active_register_df = register_df[active_companies_mask]
    active_regcodes = set(active_register_df['regcode'])
    print(f"Atrasti {len(active_regcodes)} aktīvi uzņēmumi.")

    print("Veic attiecību priekšapstrādi...")
    links_df = members_df[members_df['legal_entity_registration_number'].isin(active_regcodes) & members_df['at_legal_entity_registration_number'].isin(active_regcodes)].copy()
    source_nodes = set(links_df['legal_entity_registration_number'])
    target_nodes = set(links_df['at_legal_entity_registration_number'])
    nodes_in_structures = source_nodes.union(target_nodes)
    
    if not nodes_in_structures:
        print("Brīdinājums: Nav atrasta neviena 'mātes-meitas' attiecība starp aktīvajiem uzņēmumiem.")
        graph_data = {"nodes": [], "links": []}
    else:
        print(f"Atrasti {len(nodes_in_structures)} unikāli uzņēmumi aktīvās struktūrās.")
        num_processes = os.cpu_count()
        print(f"Sāk paralēlu peļņas aprēķinu, izmantojot {num_processes} kodolus...")
        with multiprocessing.Pool(processes=num_processes, initializer=init_worker, initargs=(fs_df, is_df)) as pool:
            profit_results = {}
            total = len(nodes_in_structures)
            for i, (regcode, profit) in enumerate(pool.imap_unordered(get_latest_profit_worker, nodes_in_structures), 1):
                profit_results[regcode] = profit
                print(f"\rApstrādā peļņu: [{i}/{total}]", end="")
        print("\nPeļņas aprēķins pabeigts.")
        print("Sagatavo gala datu struktūru...")
        final_nodes_df = active_register_df[active_register_df['regcode'].isin(nodes_in_structures)].copy()
        final_nodes_df['profit'] = final_nodes_df['regcode'].map(profit_results)
        print("Saīsina uzņēmumu nosaukumus...")
        final_nodes_df['name'] = final_nodes_df['name'].str.replace("Sabiedrība ar ierobežotu atbildību", "SIA", regex=False).str.replace("Akciju sabiedrība", "AS", regex=False)
        nodes = final_nodes_df[['regcode', 'name', 'profit']].rename(columns={'regcode': 'id'}).to_dict('records')
        links = links_df.rename(columns={'legal_entity_registration_number': 'source', 'at_legal_entity_registration_number': 'target'})[['source', 'target']].to_dict('records')
        graph_data = {"nodes": nodes, "links": links}

    output_filename = "struktura.php"
    print(f"Ģenerē gala failu '{output_filename}'...")
    
    try:
        for node in graph_data.get('nodes', []):
            if 'profit' in node and isinstance(node['profit'], float) and math.isnan(node['profit']):
                node['profit'] = None
        env = Environment(loader=FileSystemLoader(CURRENT_DIR / 'templates'))
        template = env.get_template('struktura_template.html')
        html_content = template.render(graph_data_json=json.dumps(graph_data, indent=2, ensure_ascii=False))

        OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
        output_file_path = OUTPUT_DIR / output_filename
        with open(output_file_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
        print(f"\nVeiksmīgi izveidots fails: {output_file_path.resolve()}")

        # Kopējam 'assets' direktoriju
        if ASSETS_SRC_DIR.exists():
            if ASSETS_DEST_DIR.exists():
                shutil.rmtree(ASSETS_DEST_DIR)
            
            shutil.copytree(ASSETS_SRC_DIR, ASSETS_DEST_DIR)
            print(f"Veiksmīgi nokopēta 'assets' direktorija uz: {ASSETS_DEST_DIR.resolve()}")
        else:
            print(f"BRĪDINĀJUMS: 'assets' direktorija nav atrasta ceļā '{ASSETS_SRC_DIR}'. Tā netika nokopēta.")

    except Exception as e:
        print(f"\nKĻŪDA ģenerējot failu vai kopējot direktoriju: {e}")

if __name__ == '__main__':
    multiprocessing.freeze_support()
    main()